<?php

/**
 * Plugin Name: Agrion Theme Addon
 * Description: Required plugin for Agrion Theme.
 * Plugin URI:  https://layerdrops.com/
 * Version:     1.1
 * Author:      Layerdrops
 * Author URI:  https://layerdrops.com/
 * Text Domain: agrion-addon
 */

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}

require_once __DIR__ . '/vendor/autoload.php';


/**
 * Main Agrion Theme Addon Class
 *
 * The main class that initiates and runs the plugin.
 *
 * @since 1.0.0
 */
final class Agrion_Addon_Extension
{

    /**
     * Plugin Version
     *
     * @since 1.0.0
     *
     * @var string The plugin version.
     */
    const VERSION = '1.0.0';

    /**
     * Minimum Elementor Version
     *
     * @since 1.0.0
     *
     * @var string Minimum Elementor version required to run the plugin.
     */
    const MINIMUM_ELEMENTOR_VERSION = '2.0.0';

    /**
     * Minimum PHP Version
     *
     * @since 1.0.0
     *
     * @var string Minimum PHP version required to run the plugin.
     */
    const MINIMUM_PHP_VERSION = '7.0';

    /**
     * Instance
     *
     * @since 1.0.0
     *
     * @access private
     * @static
     *
     * @var Agrion_Addon_Extension The single instance of the class.
     */
    private static $_instance = null;

    /**
     * Instance
     *
     * Ensures only one instance of the class is loaded or can be loaded.
     *
     * @since 1.0.0
     *
     * @access public
     * @static
     *
     * @return Agrion_Addon_Extension An instance of the class.
     */
    public static function instance()
    {

        if (is_null(self::$_instance)) {
            self::$_instance = new self();
        }
        return self::$_instance;
    }

    /**
     * Constructor
     *
     * @since 1.0.0
     *
     * @access public
     */
    public function __construct()
    {
        $this->define_constants();
        $this->theme_fallback();

        add_action('plugins_loaded', [$this, 'on_plugins_loaded']);
    }

    /**
     * Define the required plugin constants
     *
     * @return void
     */
    public function define_constants()
    {
        define('AGRION_ADDON_VERSION', self::VERSION);
        define('AGRION_ADDON_FILE', __FILE__);
        define('AGRION_ADDON_PATH', __DIR__);
        define('AGRION_ADDON_URL', plugins_url('', AGRION_ADDON_FILE));
        define('AGRION_ADDON_ASSETS', AGRION_ADDON_URL . '/assets');
    }

    /**
     * register fallback theme functions
     *
     * @return void
     */
    public function theme_fallback()
    {
        include AGRION_ADDON_PATH . '/common/functions.php';
    }

    /**
     * Load Textdomain
     *
     * Load plugin localization files.
     *
     * Fired by `init` action hook.
     *
     * @since 1.0.0
     *
     * @access public
     */
    public function i18n()
    {

        load_plugin_textdomain('agrion-addon', false, AGRION_ADDON_PATH . '/languages');
    }

    /**
     * On Plugins Loaded
     *
     * Checks if Elementor has loaded, and performs some compatibility checks.
     * If All checks pass, inits the plugin.
     *
     * Fired by `plugins_loaded` action hook.
     *
     * @since 1.0.0
     *
     * @access public
     */
    public function on_plugins_loaded()
    {
        new Layerdrops\Agrion\Assets();
        new Layerdrops\Agrion\PostTypes();
        new Layerdrops\Agrion\Utility();
        new Layerdrops\Agrion\Megamenu();
        new Layerdrops\Agrion\Customizer();
        new Layerdrops\Agrion\Frontend\Shortcodes();


        if (is_admin()) {
            new Layerdrops\Agrion\Admin();
        }

        add_action('wp_enqueue_scripts', [$this, 'enqueue_scripts']);
        add_action('admin_enqueue_scripts', [$this, 'admin_enqueue_scripts']);
        add_action('customize_controls_enqueue_scripts', [$this, 'customize_controls_scripts']);


        if ($this->is_compatible()) {
            add_action('elementor/init', [$this, 'init']);
        }
    }

    public function customize_controls_scripts()
    {
        wp_enqueue_script('select2');
        wp_enqueue_script('agrion-addon-customizer');
    }

    public function admin_enqueue_scripts()
    {
        wp_enqueue_style('select2');
        wp_enqueue_style('agrion-addon-admin-style');
    }

    public function enqueue_scripts()
    {
        wp_enqueue_style('animate');
        wp_enqueue_style('custom-animate');
        wp_enqueue_style('bootstrap-select');
        wp_enqueue_style('bxslider');
        wp_enqueue_style('jarallax');
        wp_enqueue_style('jquery-magnific-popup');
        wp_enqueue_style('odometer');
        wp_enqueue_style('owl-carousel');
        wp_enqueue_style('owl-theme');
        wp_enqueue_style('reey-font');
        wp_enqueue_style('swiper');
        wp_enqueue_style('agrion-icon-2');
        wp_enqueue_style('agrion-addon-style');

        wp_enqueue_script('bootstrap-select');
        wp_enqueue_script('jquery-bxslider');
        wp_enqueue_script('countdown');
        wp_enqueue_script('jarallax');
        wp_enqueue_script('jquery-ajaxchimp');
        wp_enqueue_script('jquery-appear');
        wp_enqueue_script('jquery-magnific-popup');
        wp_enqueue_script('odometer');
        wp_enqueue_script('owl-carousel');
        wp_enqueue_script('jquery-circle-progress');
        wp_enqueue_script('swiper');
        wp_enqueue_script('wow');
        wp_enqueue_script('sharer');
        wp_enqueue_script('agrion-addon-script');
    }

    /**
     * Compatibility Checks
     *
     * Checks if the installed version of Elementor meets the plugin's minimum requirement.
     * Checks if the installed PHP version meets the plugin's minimum requirement.
     *
     * @since 1.0.0
     *
     * @access public
     */
    public function is_compatible()
    {

        // Check if Elementor installed and activated
        if (!did_action('elementor/loaded')) {
            add_action('admin_notices', [$this, 'admin_notice_missing_main_plugin']);
            return false;
        }

        // Check for required Elementor version
        if (!version_compare(ELEMENTOR_VERSION, self::MINIMUM_ELEMENTOR_VERSION, '>=')) {
            add_action('admin_notices', [$this, 'admin_notice_minimum_elementor_version']);
            return false;
        }

        // Check for required PHP version
        if (version_compare(PHP_VERSION, self::MINIMUM_PHP_VERSION, '<')) {
            add_action('admin_notices', [$this, 'admin_notice_minimum_php_version']);
            return false;
        }

        return true;
    }

    /**
     * Initialize the plugin
     *
     * Load the plugin only after Elementor (and other plugins) are loaded.
     * Load the files required to run the plugin.
     *
     * Fired by `plugins_loaded` action hook.
     *
     * @since 1.0.0
     *
     * @access public
     */
    public function init()
    {

        $this->i18n();



        // register category
        add_action('elementor/elements/categories_registered', [$this, 'add_elementor_widget_categories']);
        // load icons
        add_filter('elementor/icons_manager/additional_tabs', array($this, 'add_elementor_custom_icons'));

        // Add Plugin actions
        add_action('elementor/widgets/widgets_registered', [$this, 'init_widgets']);
    }

    /**
     * Init Widgets
     *
     * Include widgets files and register them
     *
     * @since 1.0.0
     *
     * @access public
     */
    public function init_widgets()
    {

        // Register widget
        \Elementor\Plugin::instance()->widgets_manager->register(new Layerdrops\Agrion\Widgets\Header());
        \Elementor\Plugin::instance()->widgets_manager->register(new Layerdrops\Agrion\Widgets\MainSlider());
        \Elementor\Plugin::instance()->widgets_manager->register(new Layerdrops\Agrion\Widgets\About());
        \Elementor\Plugin::instance()->widgets_manager->register(new Layerdrops\Agrion\Widgets\FancyBox());
        \Elementor\Plugin::instance()->widgets_manager->register(new Layerdrops\Agrion\Widgets\IconBox());
        \Elementor\Plugin::instance()->widgets_manager->register(new Layerdrops\Agrion\Widgets\FunFact());
        \Elementor\Plugin::instance()->widgets_manager->register(new Layerdrops\Agrion\Widgets\Faq());
        \Elementor\Plugin::instance()->widgets_manager->register(new Layerdrops\Agrion\Widgets\Service());
        \Elementor\Plugin::instance()->widgets_manager->register(new Layerdrops\Agrion\Widgets\Project());
        \Elementor\Plugin::instance()->widgets_manager->register(new Layerdrops\Agrion\Widgets\Testimonials());
        \Elementor\Plugin::instance()->widgets_manager->register(new Layerdrops\Agrion\Widgets\Blog());
        \Elementor\Plugin::instance()->widgets_manager->register(new Layerdrops\Agrion\Widgets\Sponsors());
        \Elementor\Plugin::instance()->widgets_manager->register(new Layerdrops\Agrion\Widgets\CallToAction());
        \Elementor\Plugin::instance()->widgets_manager->register(new Layerdrops\Agrion\Widgets\Video());
        \Elementor\Plugin::instance()->widgets_manager->register(new Layerdrops\Agrion\Widgets\Team());
        \Elementor\Plugin::instance()->widgets_manager->register(new Layerdrops\Agrion\Widgets\ContactInfo());
        \Elementor\Plugin::instance()->widgets_manager->register(new Layerdrops\Agrion\Widgets\Offer());
        \Elementor\Plugin::instance()->widgets_manager->register(new Layerdrops\Agrion\Widgets\Gallery());
        \Elementor\Plugin::instance()->widgets_manager->register(new Layerdrops\Agrion\Widgets\ServiceDetails());
        \Elementor\Plugin::instance()->widgets_manager->register(new Layerdrops\Agrion\Widgets\Pagination());
        \Elementor\Plugin::instance()->widgets_manager->register(new Layerdrops\Agrion\Widgets\Shape());

        \Elementor\Plugin::instance()->widgets_manager->register(new Layerdrops\Agrion\Widgets\GoogleMaps());



        \Elementor\Plugin::instance()->widgets_manager->register(new Layerdrops\Agrion\Widgets\Shop());

        if (function_exists('wpcf7')) {
            \Elementor\Plugin::instance()->widgets_manager->register(new Layerdrops\Agrion\Widgets\ContactForm());
        }

        \Elementor\Plugin::instance()->widgets_manager->register(new Layerdrops\Agrion\Widgets\SideBar());

        \Elementor\Plugin::instance()->widgets_manager->register(new Layerdrops\Agrion\Widgets\FooterAbout());
        \Elementor\Plugin::instance()->widgets_manager->register(new Layerdrops\Agrion\Widgets\FooterNews());
        \Elementor\Plugin::instance()->widgets_manager->register(new Layerdrops\Agrion\Widgets\FooterNavMenu());
        \Elementor\Plugin::instance()->widgets_manager->register(new Layerdrops\Agrion\Widgets\FooterSubscribe());
        \Elementor\Plugin::instance()->widgets_manager->register(new Layerdrops\Agrion\Widgets\FooterCopyright());
    }

    public function add_elementor_widget_categories($elements_manager)
    {

        $elements_manager->add_category(
            'agrion-category',
            [
                'title' => __('Agrion Addon', 'agrion-addon'),
                'icon' => 'fa fa-plug',
            ]
        );
    }

    public function add_elementor_custom_icons($array)
    {

        return array(
            'agrion' => array(
                'name'          => 'agrion',
                'label'         => 'Agrion Icons',
                'url'           => '',
                'enqueue'       => array(
                    AGRION_ADDON_URL . '/assets/vendors/agrion-icons/style.css',
                ),
                'prefix'        => '',
                'displayPrefix' => '',
                'labelIcon'     => 'icon-dairy-products',
                'ver'           => '1.1',
                'fetchJson'     => AGRION_ADDON_URL . '/assets/vendors/agrion-icons/agrion-icons-new.json',
                'native'        => 1,
            ),
        );
    }


    /**
     * Admin notice
     *
     * Warning when the site doesn't have Elementor installed or activated.
     *
     * @since 1.0.0
     *
     * @access public
     */
    public function admin_notice_missing_main_plugin()
    {

        if (isset($_GET['activate'])) unset($_GET['activate']);

        $message = sprintf(
            /* translators: 1: Plugin name 2: Elementor */
            esc_html__('"%1$s" requires "%2$s" to be installed and activated.', 'agrion-addon'),
            '<strong>' . esc_html__('Agrion Theme Addon', 'agrion-addon') . '</strong>',
            '<strong>' . esc_html__('Elementor', 'agrion-addon') . '</strong>'
        );

        printf('<div class="notice notice-warning is-dismissible"><p>%1$s</p></div>', $message);
    }

    /**
     * Admin notice
     *
     * Warning when the site doesn't have a minimum required Elementor version.
     *
     * @since 1.0.0
     *
     * @access public
     */
    public function admin_notice_minimum_elementor_version()
    {

        if (isset($_GET['activate'])) unset($_GET['activate']);

        $message = sprintf(
            /* translators: 1: Plugin name 2: Elementor 3: Required Elementor version */
            esc_html__('"%1$s" requires "%2$s" version %3$s or greater.', 'agrion-addon'),
            '<strong>' . esc_html__('Agrion Theme Addon', 'agrion-addon') . '</strong>',
            '<strong>' . esc_html__('Elementor', 'agrion-addon') . '</strong>',
            self::MINIMUM_ELEMENTOR_VERSION
        );

        printf('<div class="notice notice-warning is-dismissible"><p>%1$s</p></div>', $message);
    }

    /**
     * Admin notice
     *
     * Warning when the site doesn't have a minimum required PHP version.
     *
     * @since 1.0.0
     *
     * @access public
     */
    public function admin_notice_minimum_php_version()
    {

        if (isset($_GET['activate'])) unset($_GET['activate']);

        $message = sprintf(
            /* translators: 1: Plugin name 2: PHP 3: Required PHP version */
            esc_html__('"%1$s" requires "%2$s" version %3$s or greater.', 'agrion-addon'),
            '<strong>' . esc_html__('Agrion Theme Addon', 'agrion-addon') . '</strong>',
            '<strong>' . esc_html__('PHP', 'agrion-addon') . '</strong>',
            self::MINIMUM_PHP_VERSION
        );

        printf('<div class="notice notice-warning is-dismissible"><p>%1$s</p></div>', $message);
    }
}

Agrion_Addon_Extension::instance();
